# 📁 Project Structure

```
crypto-ai-trader/
│
├── 📄 README.md                    # Main documentation
├── 📄 QUICKSTART.md                # 5-minute setup guide
├── 📄 .env.example                 # Environment variables template
├── 📄 .gitignore                   # Git ignore rules
├── 🐳 docker-compose.yml           # Docker orchestration
│
├── 🗄️ database/
│   └── init.sql                    # Database schema & initial data
│
├── 🐍 backend/                     # Python FastAPI backend
│   ├── Dockerfile
│   ├── requirements.txt            # Python dependencies
│   │
│   └── app/
│       ├── __init__.py
│       ├── main.py                 # FastAPI application entry point
│       ├── config.py               # Configuration management
│       ├── database.py             # Database connection & session
│       ├── models.py               # SQLAlchemy ORM models
│       │
│       ├── services/               # Business logic services
│       │   ├── __init__.py
│       │   ├── market_data.py      # CCXT market data streaming
│       │   ├── ai_strategy.py      # Claude AI trading engine
│       │   ├── execution.py        # Order execution simulator
│       │   └── portfolio.py        # Portfolio state management
│       │
│       ├── api/                    # API routes & WebSocket
│       │   ├── __init__.py
│       │   ├── routes.py           # REST API endpoints
│       │   └── websocket.py        # WebSocket real-time updates
│       │
│       └── utils/                  # Utility functions
│           ├── __init__.py
│           └── indicators.py       # Technical indicators (RSI, EMA, etc.)
│
└── ⚛️ frontend/                    # React + TypeScript frontend
    ├── Dockerfile
    ├── nginx.conf                  # NGINX configuration
    ├── package.json                # Node dependencies
    ├── tsconfig.json               # TypeScript configuration
    ├── tsconfig.node.json
    ├── vite.config.ts              # Vite build configuration
    ├── index.html                  # HTML entry point
    │
    └── src/
        ├── main.tsx                # React entry point
        ├── App.tsx                 # Main application component
        └── App.css                 # Styles

```

## 🔧 Core Components

### Backend Services

| Service | File | Purpose |
|---------|------|---------|
| **Market Data** | `market_data.py` | Real-time price streaming via CCXT Pro |
| **AI Strategy** | `ai_strategy.py` | Claude API integration for decisions |
| **Execution** | `execution.py` | Paper trading simulator with fees/slippage |
| **Portfolio** | `portfolio.py` | Position tracking, PnL calculation |

### API Endpoints

All exposed via `routes.py`:

- `GET /api/portfolio` - Current portfolio state
- `GET /api/positions` - Open positions
- `GET /api/trades` - Trade history
- `GET /api/signals` - AI signal history
- `POST /api/strategy/analyze` - Trigger manual analysis
- `POST /api/orders/manual` - Place manual order
- `WS /ws` - Real-time updates

### Database Tables

Defined in `init.sql`:

- `symbols` - Trading pair definitions
- `candles` - OHLCV market data (TimescaleDB)
- `trades` - Executed trades
- `positions` - Open positions
- `ai_signals` - AI decision log
- `portfolio_snapshots` - Portfolio history
- `orders` - Order history
- `market_snapshots` - Indicator cache
- `system_logs` - Application logs

## 🎨 Frontend Components

### Main App (`App.tsx`)

- Portfolio summary cards
- AI signal display
- Open positions table
- Recent trades table
- Real-time WebSocket updates
- Manual controls

### Styling (`App.css`)

- Dark theme optimized for trading
- Responsive grid layout
- Color-coded P&L indicators
- Animated status indicators
- Mobile-friendly

## 🔄 Data Flow

```
1. Market Data Service
   ↓
2. Fetch OHLCV + Calculate Indicators
   ↓
3. Build Context for AI
   ↓
4. Claude API → Decision
   ↓
5. Validate Decision
   ↓
6. Execute Order (if valid)
   ↓
7. Update Portfolio
   ↓
8. Broadcast to Frontend (WebSocket)
   ↓
9. Update Dashboard
```

## 🐳 Docker Services

| Service | Image | Port | Purpose |
|---------|-------|------|---------|
| **db** | timescale/timescaledb | 5432 | PostgreSQL + TimescaleDB |
| **backend** | Custom (Python) | 8000 | FastAPI application |
| **frontend** | Custom (Node + NGINX) | 3000 | React SPA |

## 📊 Key Files Explained

### `config.py`
Loads all environment variables, provides typed settings throughout app

### `main.py`
- FastAPI app initialization
- Lifespan management (startup/shutdown)
- Background AI trading loop
- WebSocket connection manager

### `ai_strategy.py`
- Constructs context for Claude
- Sends API request with market data
- Parses JSON response
- Validates decisions against constraints

### `execution.py`
- Simulates market orders
- Applies fees and slippage
- Updates positions
- Calculates realized P&L

### `portfolio.py`
- Tracks cash and positions
- Calculates unrealized P&L
- Enforces position limits
- Saves snapshots

### `indicators.py`
- RSI calculation
- EMA calculation
- Volatility metrics
- Volume analysis

## 🔐 Environment Variables

Critical settings in `.env`:

```bash
# Required
ANTHROPIC_API_KEY=          # Claude API key

# Trading
PAPER_TRADING=true          # Safety flag
INITIAL_BALANCE_USDT=10000  # Starting capital

# AI
AI_CHECK_INTERVAL_SECONDS=60
AI_MIN_CONFIDENCE=0.70

# Risk Management
MAX_POSITION_SIZE_PCT=0.30
RISK_PER_TRADE_PCT=0.02
MAX_OPEN_POSITIONS=5
```

## 📝 Code Statistics

- **Backend**: ~2,000 lines Python
- **Frontend**: ~500 lines React/TypeScript
- **Database**: ~300 lines SQL
- **Config**: ~200 lines YAML/JSON
- **Total**: ~3,000 lines

All well-commented and modular!

## 🧩 Extension Points

Easy to extend:

1. **New Exchanges**: Add to `market_data.py`
2. **New Indicators**: Add to `indicators.py`
3. **Custom Strategies**: Modify `ai_strategy.py` prompts
4. **New UI Components**: Add to `App.tsx`
5. **Additional Tables**: Extend `init.sql` and `models.py`

## 🔜 Future Enhancements

Potential additions:

- Backtesting module
- Multiple AI strategies
- Arbitrage engine
- Advanced charting (TradingView integration)
- Risk analytics dashboard
- Performance reports
- Email/Telegram alerts
- Live trading mode

---

This structure keeps the codebase clean, modular, and easy to understand!
