"""
WebSocket handler for real-time updates
"""
from fastapi import WebSocket, WebSocketDisconnect
from typing import List
import json
import asyncio
import logging

logger = logging.getLogger(__name__)


class ConnectionManager:
    """Manages WebSocket connections"""
    
    def __init__(self):
        self.active_connections: List[WebSocket] = []
    
    async def connect(self, websocket: WebSocket):
        """Accept and add new connection"""
        await websocket.accept()
        self.active_connections.append(websocket)
        logger.info(f"Client connected. Total connections: {len(self.active_connections)}")
    
    def disconnect(self, websocket: WebSocket):
        """Remove connection"""
        if websocket in self.active_connections:
            self.active_connections.remove(websocket)
        logger.info(f"Client disconnected. Total connections: {len(self.active_connections)}")
    
    async def broadcast(self, message: dict):
        """Broadcast message to all connected clients"""
        if not self.active_connections:
            return
        
        message_json = json.dumps(message)
        
        # Remove dead connections while broadcasting
        dead_connections = []
        
        for connection in self.active_connections:
            try:
                await connection.send_text(message_json)
            except Exception as e:
                logger.error(f"Error sending to client: {e}")
                dead_connections.append(connection)
        
        # Clean up dead connections
        for conn in dead_connections:
            self.disconnect(conn)
    
    async def send_personal(self, message: dict, websocket: WebSocket):
        """Send message to specific client"""
        try:
            await websocket.send_text(json.dumps(message))
        except Exception as e:
            logger.error(f"Error sending personal message: {e}")


# Global connection manager
manager = ConnectionManager()


async def websocket_endpoint(websocket: WebSocket):
    """
    WebSocket endpoint for real-time updates
    
    Clients receive:
    - ticker updates
    - AI signals
    - trade executions
    - portfolio updates
    """
    await manager.connect(websocket)
    
    try:
        # Send welcome message
        await manager.send_personal({
            'type': 'connection',
            'status': 'connected',
            'message': 'Connected to Crypto AI Trader'
        }, websocket)
        
        # Keep connection alive and handle incoming messages
        while True:
            try:
                # Wait for messages from client (e.g., commands)
                data = await websocket.receive_text()
                
                # Echo back for now (can add command handling later)
                message = json.loads(data)
                logger.info(f"Received message: {message}")
                
                await manager.send_personal({
                    'type': 'ack',
                    'message': 'Message received'
                }, websocket)
                
            except WebSocketDisconnect:
                manager.disconnect(websocket)
                break
            except json.JSONDecodeError:
                await manager.send_personal({
                    'type': 'error',
                    'message': 'Invalid JSON'
                }, websocket)
            except Exception as e:
                logger.error(f"Error in websocket loop: {e}")
                break
                
    except Exception as e:
        logger.error(f"WebSocket error: {e}")
    finally:
        manager.disconnect(websocket)
