"""
SQLAlchemy models for database tables
"""
from sqlalchemy import Column, Integer, String, Numeric, Boolean, DateTime, Text, JSON, CheckConstraint
from sqlalchemy.sql import func
from datetime import datetime
from app.database import Base


class Symbol(Base):
    __tablename__ = "symbols"
    
    id = Column(Integer, primary_key=True, index=True)
    exchange = Column(String(50), nullable=False)
    symbol = Column(String(50), nullable=False)
    base = Column(String(20), nullable=False)
    quote = Column(String(20), nullable=False)
    tick_size = Column(Numeric(20, 10))
    lot_size = Column(Numeric(20, 10))
    min_notional = Column(Numeric(20, 10))
    is_active = Column(Boolean, default=True)
    created_at = Column(DateTime(timezone=True), server_default=func.now())


class Candle(Base):
    __tablename__ = "candles"
    
    timestamp = Column(DateTime(timezone=True), primary_key=True)
    exchange = Column(String(50), primary_key=True)
    symbol = Column(String(50), primary_key=True)
    timeframe = Column(String(10), primary_key=True)
    open = Column(Numeric(20, 8), nullable=False)
    high = Column(Numeric(20, 8), nullable=False)
    low = Column(Numeric(20, 8), nullable=False)
    close = Column(Numeric(20, 8), nullable=False)
    volume = Column(Numeric(20, 8), nullable=False)


class Trade(Base):
    __tablename__ = "trades"
    
    id = Column(Integer, primary_key=True, index=True)
    timestamp = Column(DateTime(timezone=True), nullable=False, default=func.now())
    exchange = Column(String(50), nullable=False)
    symbol = Column(String(50), nullable=False)
    side = Column(String(10), nullable=False)
    type = Column(String(20), nullable=False)
    quantity = Column(Numeric(20, 8), nullable=False)
    price = Column(Numeric(20, 8), nullable=False)
    fee = Column(Numeric(20, 8), nullable=False, default=0)
    fee_currency = Column(String(10))
    pnl_realized = Column(Numeric(20, 8))
    order_id = Column(String(100))
    ai_signal_id = Column(Integer)
    metadata = Column(JSON)
    
    __table_args__ = (
        CheckConstraint("side IN ('BUY', 'SELL')"),
        CheckConstraint("type IN ('market', 'limit')"),
    )


class Position(Base):
    __tablename__ = "positions"
    
    id = Column(Integer, primary_key=True, index=True)
    symbol = Column(String(50), nullable=False, unique=True)
    exchange = Column(String(50), nullable=False)
    side = Column(String(10), nullable=False)
    quantity = Column(Numeric(20, 8), nullable=False)
    entry_price = Column(Numeric(20, 8), nullable=False)
    current_price = Column(Numeric(20, 8))
    leverage = Column(Integer, default=1)
    pnl_unrealized = Column(Numeric(20, 8), default=0)
    pnl_realized = Column(Numeric(20, 8), default=0)
    stop_loss = Column(Numeric(20, 8))
    take_profit = Column(Numeric(20, 8))
    liquidation_price = Column(Numeric(20, 8))
    opened_at = Column(DateTime(timezone=True), nullable=False, default=func.now())
    updated_at = Column(DateTime(timezone=True), nullable=False, default=func.now(), onupdate=func.now())
    
    __table_args__ = (
        CheckConstraint("side IN ('LONG', 'SHORT')"),
    )


class AISignal(Base):
    __tablename__ = "ai_signals"
    
    id = Column(Integer, primary_key=True, index=True)
    timestamp = Column(DateTime(timezone=True), nullable=False, default=func.now())
    symbol = Column(String(50), nullable=False)
    action = Column(String(10), nullable=False)
    confidence = Column(Numeric(5, 4), nullable=False)
    size_usdt = Column(Numeric(20, 8))
    reasoning = Column(Text, nullable=False)
    context = Column(JSON, nullable=False)
    stop_loss = Column(Numeric(20, 8))
    take_profit = Column(Numeric(20, 8))
    executed = Column(Boolean, default=False)
    execution_id = Column(Integer)
    
    __table_args__ = (
        CheckConstraint("action IN ('BUY', 'SELL', 'HOLD', 'CLOSE')"),
        CheckConstraint("confidence >= 0 AND confidence <= 1"),
    )


class PortfolioSnapshot(Base):
    __tablename__ = "portfolio_snapshots"
    
    id = Column(Integer, primary_key=True, index=True)
    timestamp = Column(DateTime(timezone=True), nullable=False, default=func.now())
    equity_usdt = Column(Numeric(20, 8), nullable=False)
    cash_usdt = Column(Numeric(20, 8), nullable=False)
    positions_value_usdt = Column(Numeric(20, 8), nullable=False)
    pnl_realized_total = Column(Numeric(20, 8), nullable=False)
    pnl_unrealized_total = Column(Numeric(20, 8), nullable=False)
    pnl_total = Column(Numeric(20, 8), nullable=False)
    pnl_pct = Column(Numeric(10, 6), nullable=False)
    num_positions = Column(Integer, nullable=False)
    num_trades = Column(Integer, nullable=False)
    drawdown_pct = Column(Numeric(10, 6))
    metadata = Column(JSON)


class Order(Base):
    __tablename__ = "orders"
    
    id = Column(Integer, primary_key=True, index=True)
    order_id = Column(String(100), nullable=False, unique=True)
    timestamp = Column(DateTime(timezone=True), nullable=False, default=func.now())
    exchange = Column(String(50), nullable=False)
    symbol = Column(String(50), nullable=False)
    side = Column(String(10), nullable=False)
    type = Column(String(20), nullable=False)
    quantity = Column(Numeric(20, 8), nullable=False)
    price = Column(Numeric(20, 8))
    status = Column(String(20), nullable=False)
    filled_quantity = Column(Numeric(20, 8), default=0)
    avg_fill_price = Column(Numeric(20, 8))
    fee = Column(Numeric(20, 8), default=0)
    ai_signal_id = Column(Integer)
    metadata = Column(JSON)
    updated_at = Column(DateTime(timezone=True), nullable=False, default=func.now(), onupdate=func.now())
    
    __table_args__ = (
        CheckConstraint("side IN ('BUY', 'SELL')"),
        CheckConstraint("type IN ('market', 'limit', 'stop_loss', 'take_profit')"),
        CheckConstraint("status IN ('pending', 'filled', 'partially_filled', 'cancelled', 'rejected')"),
    )


class MarketSnapshot(Base):
    __tablename__ = "market_snapshots"
    
    id = Column(Integer, primary_key=True, index=True)
    timestamp = Column(DateTime(timezone=True), nullable=False, default=func.now())
    exchange = Column(String(50), nullable=False)
    symbol = Column(String(50), nullable=False)
    indicators = Column(JSON, nullable=False)


class SystemLog(Base):
    __tablename__ = "system_logs"
    
    id = Column(Integer, primary_key=True, index=True)
    timestamp = Column(DateTime(timezone=True), nullable=False, default=func.now())
    level = Column(String(20), nullable=False)
    component = Column(String(100), nullable=False)
    message = Column(Text, nullable=False)
    metadata = Column(JSON)
    
    __table_args__ = (
        CheckConstraint("level IN ('DEBUG', 'INFO', 'WARNING', 'ERROR', 'CRITICAL')"),
    )
